<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login");
    exit();
}

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../config/config.php';

use App\Model\BlogManager;

$blogManager = new BlogManager($conn);

function processContentImages($content) {
    $dom = new DOMDocument();
    libxml_use_internal_errors(true);
    // Hack to load HTML with UTF-8 encoding
    $dom->loadHTML(mb_encode_numericentity($content, [0x80, 0x10FFFF, 0, ~0], 'UTF-8'), LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();
    
    $images = $dom->getElementsByTagName('img');
    $processed_content = $content;
    
    // Loop through images backwards (since we're modifying the array)
    for ($i = $images->length - 1; $i >= 0; $i--) {
        $img = $images->item($i);
        $src = $img->getAttribute('src');
        
        // Check if the image is base64 encoded
        if (strpos($src, 'data:image') === 0) {
            // Extract image information
            $image_parts = explode(';base64,', $src);
            $image_type_aux = explode('image/', $image_parts[0]);
            $image_type = $image_type_aux[1];
            $image_base64 = $image_parts[1];
            
            // Generate unique filename
            $upload_dir = 'assets/content/';
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $file_name = uniqid() . '.' . $image_type;
            $file_path = $upload_dir . $file_name;
            
            // Save image file
            file_put_contents($file_path, base64_decode($image_base64));
            
            // Replace base64 image with new file path in content
            $processed_content = str_replace($src, $file_path, $processed_content);
        }
    }
    
    return $processed_content;
}

ini_set('upload_max_filesize', '10M');
ini_set('post_max_size', '10M');

// Image resize function
function resizeAndSaveImage($sourceFile, $targetPath, $maxWidth = 800, $maxHeight = 600) {
    list($width, $height, $type) = getimagesize($sourceFile);
    
    // Calculate new dimensions
    $ratio = min($maxWidth / $width, $maxHeight / $height);
    $new_width = round($width * $ratio);
    $new_height = round($height * $ratio);
    
    // Create new image
    $new_image = imagecreatetruecolor($new_width, $new_height);
    
    // Handle PNG transparency
    if($type == IMAGETYPE_PNG) {
        imagecolortransparent($new_image, imagecolorallocate($new_image, 0, 0, 0));
        imagealphablending($new_image, false);
        imagesavealpha($new_image, true);
    }
    
    // Load source image
    switch($type) {
        case IMAGETYPE_JPEG:
            $source = imagecreatefromjpeg($sourceFile);
            break;
        case IMAGETYPE_PNG:
            $source = imagecreatefrompng($sourceFile);
            break;
        case IMAGETYPE_GIF:
            $source = imagecreatefromgif($sourceFile);
            break;
        default:
            return false;
    }
    
    // Resize image
    imagecopyresampled($new_image, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
    
    // Save image
    switch($type) {
        case IMAGETYPE_JPEG:
            imagejpeg($new_image, $targetPath, 90);
            break;
        case IMAGETYPE_PNG:
            imagepng($new_image, $targetPath, 9);
            break;
        case IMAGETYPE_GIF:
            imagegif($new_image, $targetPath);
            break;
    }
    
    imagedestroy($new_image);
    imagedestroy($source);
    
    return true;
}

// Handle blog operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $title = $_POST['title'];
                $content = processContentImages($_POST['content']); // Process content before saving
                $author = $_POST['author'];
                
                // Handle image upload
                $featured_image = '';
                if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === 0) {
                    $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                    $max_size = 5 * 1024 * 1024; // 5MB
                    
                    $file_type = $_FILES['featured_image']['type'];
                    $file_size = $_FILES['featured_image']['size'];
                    
                    if (!in_array($file_type, $allowed_types)) {
                        die('File type not allowed');
                    }
                    
                    if ($file_size > $max_size) {
                        die('File too large');
                    }
                    
                    $upload_dir = 'assets/img/';
                    if (!file_exists($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    
                    $file_extension = pathinfo($_FILES['featured_image']['name'], PATHINFO_EXTENSION);
                    $unique_filename = uniqid() . '.' . $file_extension;
                    $featured_image = $upload_dir . $unique_filename;
                    
                    if (!resizeAndSaveImage($_FILES['featured_image']['tmp_name'], $featured_image)) {
                        die('Failed to process image');
                    }
                }
                
                $stmt = $conn->prepare("INSERT INTO blogs (title, content, author, featured_image, publish_date, status) VALUES (?, ?, ?, ?, ?, 'draft')");
                $stmt->execute([$title, $content, $author, $featured_image, $_POST['publish_date']]);
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
                break;

            case 'update':
                $id = $_POST['blog_id'];
                $title = $_POST['title'];
                $content = processContentImages($_POST['content']); // Process content before saving
                $author = $_POST['author'];
                
                $featured_image = $_POST['current_image'];
                if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === 0) {
                    $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                    $max_size = 5 * 1024 * 1024;
                    
                    $file_type = $_FILES['featured_image']['type'];
                    $file_size = $_FILES['featured_image']['size'];
                    
                    if (!in_array($file_type, $allowed_types)) {
                        die('File type not allowed');
                    }
                    
                    if ($file_size > $max_size) {
                        die('File too large');
                    }
                    
                    $upload_dir = 'assets/img/';
                    $file_extension = pathinfo($_FILES['featured_image']['name'], PATHINFO_EXTENSION);
                    $unique_filename = uniqid() . '.' . $file_extension;
                    $featured_image = $upload_dir . $unique_filename;
                    
                    if (!resizeAndSaveImage($_FILES['featured_image']['tmp_name'], $featured_image)) {
                        die('Failed to process image');
                    }
                    
                    // Delete old image
                    if (!empty($_POST['current_image']) && file_exists($_POST['current_image'])) {
                        unlink($_POST['current_image']);
                    }
                }
                
                $stmt = $conn->prepare("UPDATE blogs SET title = ?, content = ?, author = ?, featured_image = ?, publish_date = ? WHERE id = ?");
                $stmt->execute([$title, $content, $author, $featured_image, $_POST['publish_date'], $id]);
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
                break;

            case 'delete':
                $id = $_POST['blog_id'];
                $stmt = $conn->prepare("SELECT featured_image FROM blogs WHERE id = ?");
                $stmt->execute([$id]);
                $blog = $stmt->fetch();
                
                if (!empty($blog['featured_image']) && file_exists($blog['featured_image'])) {
                    unlink($blog['featured_image']);
                }
                
                $stmt = $conn->prepare("DELETE FROM blogs WHERE id = ?");
                $stmt->execute([$id]);
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
                break;

            case 'update_status':
                $id = $_POST['blog_id'];
                $status = $_POST['status'];
                $stmt = $conn->prepare("UPDATE blogs SET status = ? WHERE id = ?");
                $stmt->execute([$status, $id]);
                echo json_encode(['success' => true]);
                exit;
                break;
        }
    }
}

// Fetch blogs
$blogs = $conn->query("SELECT * FROM blogs ORDER BY created_at DESC")->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Blog Management</title>
    <link rel="stylesheet" href="<?= BASE_URL ?>/css/admin.css" />
    <script src="https://cdn.ckeditor.com/4.22.1/standard/ckeditor.js"></script>
</head>
<body>
    <?php require_once __DIR__ . '/../src/View/partials/sidebar.php'; ?>

    <div class="main-content">
        <header>
            <div class="header-content">
                <h1>Blog Management</h1>
            </div>
        </header>

        <section class="form-section">
            <button class="btn" onclick="showCreateForm()">Create New Blog</button>
        </section>

        <section class="blog-list-section">
            <h2>All Blogs (<?= count($blogs) ?> total)</h2>
            <div class="blog-list">
                <?php if (empty($blogs)): ?>
                    <div class="blog-card">
                        <p>No blogs found. Create your first blog!</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($blogs as $blog): ?>
                    <div class="blog-card">
                        <h3><?= htmlspecialchars($blog['title']) ?></h3>
                        
                        <div class="blog-meta">
                            <strong>Author:</strong> <?= htmlspecialchars($blog['author']) ?> | 
                            <strong>Date:</strong> <?= $blog['publish_date'] ?><br>
                            <strong>Created:</strong> <?= date('Y-m-d H:i', strtotime($blog['created_at'])) ?>
                            <strong>Status:</strong> 
                            <span class="status-badge status-<?= $blog['status'] ?>">
                                <?= ucfirst($blog['status']) ?>
                            </span>
                        </div>
                        
                        <p class="blog-summary">
                            <?= substr(strip_tags($blog['content']), 0, 150) ?>...
                        </p>
                        
                        <div class="card-actions">
                            <a href="blog_detail.php?id=<?= $blog['id'] ?>" target="_blank" class="btn btn-small">Read more</a>
                            <button class="btn btn-small" onclick="editBlog(<?= $blog['id'] ?>)">Edit</button>
                            <button class="btn btn-danger btn-small" onclick="deleteBlog(<?= $blog['id'] ?>)">Delete</button>
                            
                            <select onchange="updateStatus(<?= $blog['id'] ?>, this.value)" style="margin-left: 10px;" class="form-control-small">
                                <option value="draft" <?= $blog['status'] == 'draft' ? 'selected' : '' ?>>Draft</option>
                                <option value="published" <?= $blog['status'] == 'published' ? 'selected' : '' ?>>Published</option>
                            </select>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </section>
    </div>

    <!-- Create/Edit Blog Modal -->
    <div id="blogModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="hideForm()">&times;</span>
            <h2 id="modalTitle">Create New Blog</h2>
            
            <form id="blogForm" action="" method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="create">
                <input type="hidden" name="blog_id" id="blogId">
                <input type="hidden" name="current_image" id="currentImage">
                
                <div class="form-group">
                    <label for="title">Blog Title *</label>
                    <input type="text" name="title" id="title" required>
                </div>
                
                <div class="form-group">
                    <label for="author">Author *</label>
                    <input type="text" name="author" id="author" required>
                </div>
                
                <div class="form-group">
                    <label for="content">Content *</label>
                    <textarea id="content" name="content" rows="10"></textarea>
                </div>
                
                <div class="form-group">
                    <label for="featured_image">Featured Image</label>
                    <input type="file" name="featured_image" id="featured_image" accept="image/*">
                    <div id="imagePreview"></div>
                </div>
                
                <div class="form-group">
                    <label for="publish_date">Publish Date *</label>
                    <input type="date" name="publish_date" id="publish_date" required>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn">Save Blog</button>
                    <button type="button" class="btn btn-danger" onclick="hideForm()">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Initialize CKEditor
        CKEDITOR.replace('content');

        // Modal functions
        function showCreateForm() {
            document.getElementById('modalTitle').textContent = 'Create New Blog';
            document.getElementById('blogForm').reset();
            document.getElementById('imagePreview').innerHTML = '';
            document.querySelector('input[name="action"]').value = 'create';
            document.getElementById('blogModal').style.display = 'block';
            
            // Set default publish date to today
            document.getElementById('publish_date').value = new Date().toISOString().split('T')[0];
            
            // Clear CKEditor content
            if (CKEDITOR.instances.content) {
                CKEDITOR.instances.content.setData('');
            }
        }

        function hideForm() {
            document.getElementById('blogModal').style.display = 'none';
        }

        async function editBlog(id) {
            try {
                const response = await fetch(`get_blog?id=${id}`);
                if (!response.ok) {
                    throw new Error('Failed to fetch blog data');
                }
                const blog = await response.json();
                
                document.getElementById('modalTitle').textContent = 'Edit Blog';
                document.getElementById('blogId').value = blog.id;
                document.getElementById('title').value = blog.title;
                document.getElementById('author').value = blog.author;
                document.getElementById('currentImage').value = blog.featured_image || '';
                document.getElementById('publish_date').value = blog.publish_date;
                
                // Set CKEditor content
                CKEDITOR.instances.content.setData(blog.content || '');
                
                // Show current image preview
                const imagePreview = document.getElementById('imagePreview');
                if (blog.featured_image) {
                    imagePreview.innerHTML = `<img src="${blog.featured_image}" alt="Current Image" class="image-preview">`;
                } else {
                    imagePreview.innerHTML = '';
                }
                
                document.querySelector('input[name="action"]').value = 'update';
                document.getElementById('blogModal').style.display = 'block';
            } catch (error) {
                console.error('Error fetching blog:', error);
                alert('Failed to load blog data. Please try again.');
            }
        }

        function deleteBlog(id) {
            if (confirm('Are you sure you want to delete this blog? This action cannot be undone.')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="blog_id" value="${id}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        async function updateStatus(id, status) {
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=update_status&blog_id=${id}&status=${status}`
                });
                
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                
                const result = await response.json();
                if (result.success) {
                    // Refresh page to show updated status
                    location.reload();
                }
            } catch (error) {
                console.error('Error updating status:', error);
                alert('Failed to update status. Please try again.');
            }
        }

        // Image preview functionality
        document.getElementById('featured_image').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('imagePreview');
            
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.innerHTML = `<img src="${e.target.result}" alt="Preview" class="image-preview">`;
                };
                reader.readAsDataURL(file);
            } else {
                preview.innerHTML = '';
            }
        });

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('blogModal');
            if (event.target == modal) {
                hideForm();
            }
        }

        // Close modal with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                hideForm();
            }
        });
    </script>
</body>
</html>